//===============================================================================================
// Validate the user entered customer info (billing/shipping info)
//
// Copyright <c> 2003 Intuit, Inc.  All rights reserved
//===============================================================================================

// Field types
kNameType =			1;
kInitialType =		2;
kAddressType =		3;
kCityType =			4;
kStateType =		5;
kZipType =			6;
kPostalCodeType =	7;
kCityType =			8;
kPhoneType =		9;
kPhoneExtType =	10;
kEmailType =		11;
kExpMonthType =	12;
kExpYearType =		13;
kCCNumType =		14;

kMissingValue = "Please supply the fields marked in red";
kInvalidPhone = "Enter a valid phone number.";
kInvalidPhoneExt = "Phone number extension must be a number.";
kInvalidInitial = "Middle initial must be a letter.";
kInvalidAddress = "An address cannot contain '?', '@', or '&'.";
kInvalidZip = "Enter a valid ZIP code.";
kInvalidCCNum = "Enter a valid credit card number.";
kInvalidName = "A name cannot start with a space.";
kNameToSmall = "A name must have at least 2 alphabetic characters.";
kInvalidExpirationDate = "Please try another credit card because date expired.";
kMissingMonthExpirationDate = "Please select a month for your credit card expiration date.";
kMissingYearExpirationDate = "Please select a year for your credit card expiration date.";

//---------------------------------------------------------------------------------------------
// Field info struct
//---------------------------------------------------------------------------------------------
function zValidateField( id, type, bRequired, labelId )
{
	this.id = id;
	this.type = type;
	this.bRequired =( bRequired ) ? true : false;
	if( labelId ) this.labelId = labelId; 
}

//---------------------------------------------------------------------------------------------
function _zValidateFieldValidate( errList )
{
	var bValid = true;
	var errMsg = null;

	try
	{
		if( this.bRequired && this.labelId ) setCss( this.labelId, this.css );

		var bPopulated = isInputPopulated( this.id );
		if( !bPopulated )
		{
			if( this.bRequired )
			{
				// Set the correct error message
				switch( this.type )
				{
					case kExpMonthType:
						throw kMissingMonthExpirationDate;
						break;
					case kExpYearType: 
						throw kMissingYearExpirationDate;
						break;
					default:
						throw kMissingValue;
						break;						
				} // end switch												
			} // end if( this.bRequired )
			return true;
		} // end if( !bPopulated )

		// check the data for edit controls
		var e = document.getElementById( this.id );
		if(!e) throw "";
		if( e.tagName.toLowerCase() != "input" )
		{
		   // Exception case: Need to verify expiration month and year;
		   // so do not return yet.  Let continue.
		   if( e.id.toLowerCase() == "cardexpirationmonth" ||
		       e.id.toLowerCase() == "cardexpirationyear"  )
		   {
		       // let fall through to resume checking
		   }
		   else
		       return true; // leave
		}

		var val = e.value;

		switch( this.type )
		{
			case kPhoneType:
				if( !validatePhone( val ) ) throw kInvalidPhone;
				break;

			case kPhoneExtType:
				if( !validatePhoneExt( val ) ) throw kInvalidPhoneExt;
				break;

			case kNameType: 
				if( !validateCardHolderName( val ) ) throw kInvalidName;
				if (!validateMinimumCharacters( val, 2 ) ) throw kNameToSmall
				break;
			
			case kInitialType:
				if( !validateInitial( val ) ) throw kInvalidInitial;
				break;

			case kAddressType:
				if( !validateAddress( val ) ) throw kInvalidAddress;
				break;

			case kCityType: break;
			case kStateType: break;			
			
			case kZipType:
				if( !validateZip( val ) ) throw kInvalidZip;
				break;

			case kPostalCodeType: break;
			case kEmailType: break;
			case kExpMonthType:
				if( readyToTestExpirationDate( kExpMonthType, val ) )
				{
					if( !validateExpirationDate() ) throw kInvalidExpirationDate;					
				}
				break;
			case kExpYearType: 
				if( readyToTestExpirationDate( kExpYearType, val ) )
				{
					if( !validateExpirationDate() ) throw kInvalidExpirationDate;					
				}
				break;

			case kCCNumType:
				if( !validateCCNum( val ) ) throw kInvalidCCNum;
				break;
		}
	}
	catch( e )
	{
		bValid = false;
		if( this.labelId ) setCss( this.labelId, this.errCss );
		if( e.length > 0 ) errMsg = e;
	}

	// Add the error message to the error list if present
	if( errMsg && errList )
	{
		// If the same error is not already present add it
		var bAdd = true;
		var cnt = errList.length;
		
		for( var i=0; i<cnt; ++i )
		{
			if( errList[i] == errMsg )
			{
				bAdd = false;
				break;
			}
		}
		
		if( bAdd ) errList.push( errMsg );
	}	
		
	return bValid;
}

//---------------------------------------------------------------------------------------------
p = zValidateField.prototype

p.labelId = null;
p.css = "label"; // normal field label style
p.errCss = "labelErr"; // error field label style

p.validate = _zValidateFieldValidate;

p = null;


//---------------------------------------------------------------------------------------------
function validateFields( fields, errList )
{
	if( !fields ) return;
	
	var bValid = true;
	var cnt = fields.length;
	
	for( i=0; i<cnt; i++ )
	{
		// Validate each field until error
		if( !fields[i].validate( errList ) ) 
		{
			bValid = false;
			break; // stop on error
		}
	}
	
	return bValid;
}

//---------------------------------------------------------------------------------------------
function validateCardHolderName( val )
{
	if( !val ) return false;
	   
	// Check for a leading space	
	return !( val.match( /^\s.*/ ) );
}

//---------------------------------------------------------------------------------------------
function validateMinimumCharacters( val, minSize )
{
	if( !val ) return false;
	
	var sVal = val.replace( /[^[A-Za-z]/gi,"");
	return (sVal.length >= minSize);
}

//---------------------------------------------------------------------------------------------
function validatePhone( val )
{
	if( !val ) return false;
	
	// Strip non digits
	var sVal = val.replace( /[^0-9]/gi, "" );
	
	// US and Canadian phone number are the same length
	return ( sVal.length == 10 );
}

//---------------------------------------------------------------------------------------------
function validatePhoneExt( val )
{
	if( !val ) return false;
	
	// Is the extension numeric
	return !( val.match( /[^0-9]/ ) );
}

//---------------------------------------------------------------------------------------------
function validateInitial( val )
	// Check if a letter
{
	if( !val ) return false;
	
	return ( val.length == 1 && val.match( /[A-Za-z]/ ) );
}

//---------------------------------------------------------------------------------------------
function validateAddress( val )
	// Check for presence of invalid characters '?', '@', or '&'
{
	if( !val ) return false;
	
	return !( val.match( /[\?\@\&]/ ) );
}

//---------------------------------------------------------------------------------------------
function validateZip( val )
	// Check for a 5 or 9 digit US zip code
{
	if( !val ) return false;
	
	// Strip non digits
	var sVal = val.replace( /[^0-9]/gi, "" );
	var len = sVal.length;
	
	return ( 5 == len || 9 == len );
}

//---------------------------------------------------------------------------------------------
function validateCCNum( val )
	// Check for credit card number based on card type.
	// Use a callback to get the card type
{
	// strip non digits including leading spaces
	var sVal = val.replace( /[^0-9]/gi, "" );
	var len = sVal.length;
	if( 0 == len ) return false;
	
	// Use a callback to get the card type
	var cardType = getCardType();
	if( !cardType ) return false;
   
	switch( cardType )
	{
		case "Visa":			
			return ( (len==16 || len==13) && sVal.match( /^4/ ) && CheckMod10(sVal) );			
			
		case "MasterCard":
			return ( len == 16 && sVal.match( /^5[12345]/ ) && CheckMod10(sVal) );
			
		case "American Express":
			return ( len == 15 && sVal.match( /^3[47]/ ) && CheckMod10(sVal) );
			
		case "Discover":
			return ( len == 16 && sVal.match( /^6011/ ) && CheckMod10(sVal) );
	}
	
	return false;
}

//---------------------------------------------------------------------------------------------
function CheckMod10( val )
	// Validate the credit card number by doing a modulus 10 checksum.
{
	/// Below is the algorithm description provided by SPC or CTO OrderManagement team (Ron Groth or Shuping Jia):
	/// The following steps are required to validate the primary account number using the "mod 10" algorithm.	
	/// <item><description>Double the value of alternate digits of the primary account number beginning with the second digit from the right (the first right--hand digit is the check digit.)</description></item>
	/// <item><description>Add the individual digits comprising the products obtained in Step 1 to each of the unaffected digits in the original number.</description></item>
	/// <item><description>The total obtained in Step 2 must be a number ending in zero (30, 40, 50, etc.) for the account number to be validated.</description></item>
	/// <br>For example, to validate the primary account number 49927398716:</br>
	/// <br>Step 1: </br>
	/// <br>        4 9 9 2 7 3 9 8 7 1 6</br>
	/// <br>         x2  x2  x2  x2  x2 </br>
	/// <br>------------------------------</br>
	/// <br>         18   4   6  16   2</br>
	/// <br>Step 2: 4 + (1+8) + 9 + (4) + 7 + (6) + 9 + (1+6) + 7 + (2) + 6 </br>
	/// <br>Step 3: Sum = 70: Card number is validated </br>
	/// <br>Note: Card is valid because the 70/10 yields no remainder. </br>

	// Check for empty string
	var len = val.length;
	if( 0 == len ) return false;

	// Prepare the array
	var arrayVal = new Array( len ); // array of characters
	arrayVal = val.split(""); // break at every character
	arrayVal.reverse(); // reverse the order since we start w/ 2nd digit from right
	var iCheckSum = 0;
	for(var i=0; i < arrayVal.length; ++i)
	{
		if( (i % 2)==0 )
		{
			// even array element 0,2,4 etc., add to Total.
			iCheckSum += parseInt( arrayVal[i], 10 ); // convert from string to integer
		}
		else
		{
			// odd array element 1,3,5 etc., multiply by 2.  If result is >= 10, then 
			// subtract 9, then add to Total; otherwise if result < 10, then just add to
			// total.
			// CTO algorithm is:
			// Step 1: Double the value of alternate digits of the primary 
			//         account number beginning with the second digit from 
			//         the right (the first right--hand digit is the check digit.) 
			//	for (int i = (NumberLength % 2); i < NumberLength; i += 2) 
			//	{
			//		int Digit = Int32.Parse(number.Substring(i, 1)) * 2;
			//		if (Digit < 10) 
			//		{
			//			Checksum += Digit;
			//		} 
			//		else 
			//		{
			//			Checksum += Digit - 9;
			//		}
			//	}
			var iTmp = 0;
			iTmp = parseInt(arrayVal[i],10) * 2; // convert from string to integer; double result
			if( iTmp >= 10 )
				iTmp = iTmp - 9;								
			iCheckSum += iTmp;
		}
	}

	// Mod test
	var vModDivisor = 10;
	
	return ( (iCheckSum % vModDivisor)==0 );
}

//---------------------------------------------------------------------------------------------
//
//	Function object: zExpirationDateClass and static members
//
// Create and initialize the "static" variables specific for
// function object zExpirationDateClass.
// Function declarations are processed before code is executed, so
// we really can do the below assignment before the function declaration.
// This is needed because the month and year are processed in a piecemeal
// fashion and zExpirationDateClass will store this information.  It then
// can be used for comparision when all information is complete.  This is
// better than using global variables because the static variables are 
// limited to the zExpirationDateClass function object.
zExpirationDateClass.month = "";
zExpirationDateClass.year = "";
zExpirationDateClass.dateComplete = false;
zExpirationDateClass.expDate = "";

// Function object zExpirationDateClass
function zExpirationDateClass( month, year )
{
	if( month != "" )
		zExpirationDateClass.month = month;
	if( year != "" )
		zExpirationDateClass.year = year;
		
	if( zExpirationDateClass.month != "" && zExpirationDateClass.year != "" )
	{
		// Set flag that expiration date complete
		zExpirationDateClass.dateComplete = true;
		
		// Create a date
		tmpDate = new Date();
		tmpDate.setMonth(zExpirationDateClass.month - 1); // month is zero based
		tmpDate.setDate(1); // default to first of month
		tmpDate.setFullYear(zExpirationDateClass.year);
		zExpirationDateClass.expDate = tmpDate;
	}
}

// Helper functions for zExpirationDateClass
function getUserExpirationDate()
{
	return zExpirationDateClass.expDate;
}

function resetExpirationDateClass()
{
	zExpirationDateClass.month = "";
	zExpirationDateClass.year = "";
	zExpirationDateClass.dateComplete = false;
	zExpirationDateClass.expDate = "";	
}

//
//	End of Function object: zExpirationDateClass and static members
//

//---------------------------------------------------------------------------------------------
function readyToTestExpirationDate( valType, val )
	// Add the expiration month and year to function object zExpirationDateClass
	// so all the pieces are collected to compare date with current system date.
{
	var bReadyToTest = false;
	
	switch( valType )
	{
		case kExpMonthType:
			zExpirationDateClass( val, "" );
			break;
		case kExpYearType:
			zExpirationDateClass( "", val );
			break;
	}
	
	if( zExpirationDateClass.dateComplete )
		bReadyToTest = true;
	
	return bReadyToTest;
}

//---------------------------------------------------------------------------------------------
function validateExpirationDate()
{
	var bValid = true;
	
	if( zExpirationDateClass.dateComplete == true )
	{
		now = new Date(); // current system time
		
		// Get user entered expiration date
		testDate = getUserExpirationDate();
		
		if( now.getTime() > testDate.getTime() )
			bValid = false; // expired
	}
	
	// Reset the zExpirationDateClass members because
	// user may be on the "Provide Your Credit Card Information"
	// screen (i.e. payOnline.htm) fixing an error.  If we don't
	// reset zExpirationDateClass, then we will use the previous values
	// that were first entered.
	resetExpirationDateClass();
	
	return bValid;
}